/*
  WS2812FX.cpp - Library for WS2812 LED effects.

  Harm Aldick - 2016
  www.aldick.org

  2017-04-10   CAL - Added new modes: OFF and AUTO
  2020-12-20   CAL - major changes for Crazy Light
  2021-12-23   CAL - Removed Crazy Light modes and added new modes for TriLight
*/

#include "WS2812FX.h"

#define CALL_MODE(n) (this->*_mode[n])();
#define min(a,b) (((a)<(b))?(a):(b))

void WS2812FX::init() {
  Adafruit_NeoPixel::begin();
  WS2812FX::setBrightness(_brightness);
  Adafruit_NeoPixel::show();
}

void WS2812FX::service() {

  if (_running) {

    unsigned long now = millis();

    // Turn auto mode off after an hour
    if (_auto_mode && (now > _autoTimeout)) {
      _auto_mode = false;
      _mode_index = FX_MODE_OFF;
    }

    // Begin new auto mode - CAL
    // Is auto_mode enabled ?
    if (_auto_mode && (now > _auto_mode_change_time)) {
      // Calculate next mode change time
      _auto_mode_change_time = now + AUTO_CHANGE_DELAY_MS;

      // Turn off all LEDs
      strip_off();

      // Setup to call random mode with random color, speed and brightness
      _counter_mode_call = 0;
      _counter_mode_step = 0;
      _mode_last_call_time = 0;
      _inc = 0;

      // Select a random mode but not auto or off
      do {
        _mode_index = random(MODE_COUNT);
      } while ((_mode_index == FX_MODE_AUTO) || (_mode_index == FX_MODE_OFF));

      // Select a random color
      _color = _mode_color = color_wheel(random(256));

      // Select a random brightness
      setBrightness(random(20, 50));

      // Select a random speed
      _speed = random(SPEED_MIN, SPEED_MAX);
    }
    // End new auto mode - CAL

    if (now - _mode_last_call_time > _mode_delay) {
      CALL_MODE(_mode_index);
      _counter_mode_call++;
      _mode_last_call_time = now;
    }
  }
}

void WS2812FX::start() {
  _counter_mode_call = 0;
  _counter_mode_step = 0;
  _mode_last_call_time = 0;
  _running = true;
}

void WS2812FX::stop() {
  _running = false;
  strip_off();
}

void WS2812FX::setMode(uint8_t m) {

  _auto_mode = false;		// CAL
  _auto_mode_change_time = 0;	// CAL

  _counter_mode_call = 0;
  _counter_mode_step = 0;
  _mode_last_call_time = 0;
  _mode_index = constrain(m, 0, MODE_COUNT - 1);
  _mode_color = _color;
  Adafruit_NeoPixel::setBrightness(_brightness);
  Adafruit_NeoPixel::clear();

  // CAL
  _inc = 0;
}

void WS2812FX::setSpeed(uint8_t s) {
  _counter_mode_call = 0;
  _counter_mode_step = 0;
  _mode_last_call_time = 0;
  _speed = constrain(s, SPEED_MIN, SPEED_MAX);
}

void WS2812FX::increaseSpeed(uint8_t s) {
  s = constrain(_speed + s, SPEED_MIN, SPEED_MAX);
  setSpeed(s);
}

void WS2812FX::decreaseSpeed(uint8_t s) {
  s = constrain(_speed - s, SPEED_MIN, SPEED_MAX);
  setSpeed(s);
}

void WS2812FX::setColor(uint8_t r, uint8_t g, uint8_t b) {
  setColor(((uint32_t)r << 16) | ((uint32_t)g << 8) | b);
}

void WS2812FX::setColor(uint32_t c) {
  _color = c;
  _counter_mode_call = 0;
  _counter_mode_step = 0;
  _mode_last_call_time = 0;
  _mode_color = _color;
  Adafruit_NeoPixel::setBrightness(_brightness);
}

void WS2812FX::setBrightness(uint8_t b) {
  _brightness = constrain(b, BRIGHTNESS_MIN, BRIGHTNESS_MAX);
  Adafruit_NeoPixel::setBrightness(_brightness);
  Adafruit_NeoPixel::show();
}

void WS2812FX::increaseBrightness(uint8_t s) {
  s = constrain(_brightness + s, BRIGHTNESS_MIN, BRIGHTNESS_MAX);
  setBrightness(s);
}

void WS2812FX::decreaseBrightness(uint8_t s) {
  s = constrain(_brightness - s, BRIGHTNESS_MIN, BRIGHTNESS_MAX);
  setBrightness(s);
}

boolean WS2812FX::isRunning() {
  return _running;
}

uint8_t WS2812FX::getMode(void) {
  return _mode_index;
}

uint8_t WS2812FX::getSpeed(void) {
  return _speed;
}

uint8_t WS2812FX::getBrightness(void) {
  return _brightness;
}

uint32_t WS2812FX::getColor(void) {
  return _color;
}

uint8_t WS2812FX::getModeCount(void) {
  return MODE_COUNT;
}

const char* WS2812FX::getModeName(uint8_t m) {
  if (m < MODE_COUNT) {
    return _name[m];
  } else {
    return "";
  }
}

/* #####################################################
  #
  #  Color and Blinken Functions
  #
  ##################################################### */

/*
   Turns everything off. Doh.
*/
void WS2812FX::strip_off() {
  Adafruit_NeoPixel::clear();
  Adafruit_NeoPixel::show();
}

/*
   Put a value 0 to 255 in to get a color value.
   The colours are a transition r -> g -> b -> back to r
   Inspired by the Adafruit examples.
*/
uint32_t WS2812FX::color_wheel(uint8_t pos) {
  pos = 255 - pos;
  if (pos < 85) {
    return ((uint32_t)(255 - pos * 3) << 16) | ((uint32_t)(0) << 8) | (pos * 3);
  } else if (pos < 170) {
    pos -= 85;
    return ((uint32_t)(0) << 16) | ((uint32_t)(pos * 3) << 8) | (255 - pos * 3);
  } else {
    pos -= 170;
    return ((uint32_t)(pos * 3) << 16) | ((uint32_t)(255 - pos * 3) << 8) | (0);
  }
}

/*
   Returns a new, random wheel index with a minimum distance of 42 from pos.
*/
uint8_t WS2812FX::get_random_wheel_index(uint8_t pos) {
  uint8_t r = 0;
  uint8_t x = 0;
  uint8_t y = 0;
  uint8_t d = 0;

  while (d < 42) {
    r = random(256);
    x = abs(pos - r);
    y = 255 - x;
    d = min(x, y);
  }

  return r;
}

/*
   No blinking. Just plain old static light.
*/
void WS2812FX::mode_static(void) {
  for (uint16_t i = 0; i < _led_count; i++) {
    Adafruit_NeoPixel::setPixelColor(i, _color);
    yield();
  }
  Adafruit_NeoPixel::show();

  _mode_delay = 50;
}

/*
   Lights all LEDs after each other up. Then turns them in
   that order off. Repeat.
*/
void WS2812FX::mode_color_wipe(void) {
  if (_counter_mode_step < _led_count) {
    Adafruit_NeoPixel::setPixelColor(_counter_mode_step, _color);
  } else {
    Adafruit_NeoPixel::setPixelColor(_counter_mode_step - _led_count, 0);
  }
  Adafruit_NeoPixel::show();

  _counter_mode_step = (_counter_mode_step + 1) % (_led_count * 2);

  _mode_delay = 5 + ((50 * (uint32_t)(SPEED_MAX - _speed)) / _led_count);
}


/*
   Turns all LEDs after each other to a random color.
   Then starts over with another color.
*/
void WS2812FX::mode_color_wipe_random(void) {
  if (_counter_mode_step == 0) {
    _mode_color = get_random_wheel_index(_mode_color);
  }

  Adafruit_NeoPixel::setPixelColor(_counter_mode_step, color_wheel(_mode_color));
  Adafruit_NeoPixel::show();

  _counter_mode_step = (_counter_mode_step + 1) % _led_count;

  _mode_delay = 5 + ((50 * (uint32_t)(SPEED_MAX - _speed)) / _led_count);
}


/*
   Lights all LEDs in one random color up. Then switches them
   to the next random color.
*/
void WS2812FX::mode_random_color(void) {
  _mode_color = get_random_wheel_index(_mode_color);

  for (uint16_t i = 0; i < _led_count; i++) {
    Adafruit_NeoPixel::setPixelColor(i, color_wheel(_mode_color));
    yield();
  }

  Adafruit_NeoPixel::show();
  _mode_delay = 100 + ((5000 * (uint32_t)(SPEED_MAX - _speed)) / SPEED_MAX);
}


/*
   Lights every LED in a random color. Changes all LED at the same time
   to new random colors.
*/
void WS2812FX::mode_multi_dynamic(void) {
  for (uint16_t i = 0; i < _led_count; i++) {
    Adafruit_NeoPixel::setPixelColor(i, color_wheel(random(256)));
    yield();
  }
  Adafruit_NeoPixel::show();
  _mode_delay = 100 + ((5000 * (uint32_t)(SPEED_MAX - _speed)) / SPEED_MAX);
}


/*
   Does the "standby-breathing" of well known i-Devices. Fixed Speed.
   Use mode "fade" if you like to have something similar with a different speed.
*/
void WS2812FX::mode_breath(void) {
  //                                      0    1    2   3   4   5   6    7   8   9  10  11   12   13   14   15   16    // step
  uint16_t breath_delay_steps[] =     {   7,   9,  13, 15, 16, 17, 18, 930, 19, 18, 15, 13,   9,   7,   4,   5,  10 }; // magic numbers for breathing LED
  uint8_t breath_brightness_steps[] = { 150, 125, 100, 75, 50, 25, 16,  15, 16, 25, 50, 75, 100, 125, 150, 220, 255 }; // even more magic numbers!

  if (_counter_mode_call == 0) {
    _mode_color = breath_brightness_steps[0] + 1;
  }

  uint8_t breath_brightness = _mode_color; // we use _mode_color to store the brightness

  if (_counter_mode_step < 8) {
    breath_brightness--;
  } else {
    breath_brightness++;
  }

  // update index of current delay when target brightness is reached, start over after the last step
  if (breath_brightness == breath_brightness_steps[_counter_mode_step]) {
    _counter_mode_step = (_counter_mode_step + 1) % (sizeof(breath_brightness_steps) / sizeof(uint8_t));
  }

  for (uint16_t i = 0; i < _led_count; i++) {
    Adafruit_NeoPixel::setPixelColor(i, _color);           // set all LEDs to selected color
    yield();
  }
  int b = map(breath_brightness, 0, 255, 0, _brightness);  // keep brightness below brightness set by user
  Adafruit_NeoPixel::setBrightness(b);                     // set new brightness to leds
  Adafruit_NeoPixel::show();

  _mode_color = breath_brightness;                         // we use _mode_color to store the brightness
  _mode_delay = breath_delay_steps[_counter_mode_step];
}

/*
   Cycles all LEDs at once through a rainbow.
*/
void WS2812FX::mode_rainbow(void) {
  uint32_t color = color_wheel(_counter_mode_step);
  for (uint16_t i = 0; i < _led_count; i++) {
    Adafruit_NeoPixel::setPixelColor(i, color);
    yield();
  }
  Adafruit_NeoPixel::show();

  _counter_mode_step = (_counter_mode_step + 1) % 256;

  _mode_delay = 1 + ((50 * (uint32_t)(SPEED_MAX - _speed)) / SPEED_MAX);
}


/*
   Cycles a rainbow over the entire string of LEDs.
*/
void WS2812FX::mode_rainbow_cycle(void) {
  for (uint16_t i = 0; i < _led_count; i++) {
    Adafruit_NeoPixel::setPixelColor(i, color_wheel(((i * 256 / _led_count) + _counter_mode_step) % 256));
    yield();
  }
  Adafruit_NeoPixel::show();

  _counter_mode_step = (_counter_mode_step + 1) % 256;

  _mode_delay = 1 + ((50 * (uint32_t)(SPEED_MAX - _speed)) / SPEED_MAX);
}


/*
   Theatre-style crawling lights.
   Inspired by the Adafruit examples.
*/
void WS2812FX::mode_theater_chase(void) {
  uint8_t j = _counter_mode_call % 6;
  if (j % 2 == 0) {
    for (uint16_t i = 0; i < _led_count; i = i + 3) {
      Adafruit_NeoPixel::setPixelColor(i + (j / 2), _color);
      yield();
    }
    Adafruit_NeoPixel::show();
    _mode_delay = 50 + ((500 * (uint32_t)(SPEED_MAX - _speed)) / SPEED_MAX);
  } else {
    for (uint16_t i = 0; i < _led_count; i = i + 3) {
      Adafruit_NeoPixel::setPixelColor(i + (j / 2), 0);
      yield();
    }
    _mode_delay = 1;
  }
}


/*
   Theatre-style crawling lights with rainbow effect.
   Inspired by the Adafruit examples.
*/
void WS2812FX::mode_theater_chase_rainbow(void) {
  uint8_t j = _counter_mode_call % 6;
  if (j % 2 == 0) {
    for (uint16_t i = 0; i < _led_count; i = i + 3) {
      Adafruit_NeoPixel::setPixelColor(i + (j / 2), color_wheel((i + _counter_mode_step) % 256));
      yield();
    }
    Adafruit_NeoPixel::show();
    _mode_delay = 50 + ((500 * (uint32_t)(SPEED_MAX - _speed)) / SPEED_MAX);
  } else {
    for (uint16_t i = 0; i < _led_count; i = i + 3) {
      Adafruit_NeoPixel::setPixelColor(i + (j / 2), 0);
      yield();
    }
    _mode_delay = 1;
  }
  _counter_mode_step = (_counter_mode_step + 1) % 256;
}

/*
   Running lights effect with smooth sine transition.
*/
void WS2812FX::mode_running_lights(void) {
  uint8_t r = ((_color >> 16) & 0xFF);
  uint8_t g = ((_color >> 8) & 0xFF);
  uint8_t b = (_color & 0xFF);

  for (uint16_t i = 0; i < _led_count; i++) {
    int s = (sin(i + _counter_mode_call) * 127) + 128;
    Adafruit_NeoPixel::setPixelColor(i, (((uint32_t)(r * s)) / 255), (((uint32_t)(g * s)) / 255), (((uint32_t)(b * s)) / 255));
    yield();
  }

  Adafruit_NeoPixel::show();

  _mode_delay = 35 + ((350 * (uint32_t)(SPEED_MAX - _speed)) / SPEED_MAX);
}

/*
   Blink several LEDs in random colors on, reset, repeat.
   Inspired by www.tweaking4all.com/hardware/arduino/adruino-led-strip-effects/
*/
void WS2812FX::mode_twinkle_random(void) {
  _mode_color = color_wheel(random(256));
  if (_counter_mode_step == 0) {
    strip_off();
    uint16_t min_leds = max(1, _led_count / 5); // make sure, at least one LED is on
    uint16_t max_leds = max(1, _led_count / 2); // make sure, at least one LED is on
    _counter_mode_step = random(min_leds, max_leds);
  }

  Adafruit_NeoPixel::setPixelColor(random(_led_count), _mode_color);
  Adafruit_NeoPixel::show();

  _counter_mode_step--;
  _mode_delay = 50 + ((1986 * (uint32_t)(SPEED_MAX - _speed)) / SPEED_MAX);
}


/*
   Blink several LEDs on, fading out.
*/
void WS2812FX::mode_twinkle_fade(void) {

  for (uint16_t i = 0; i < _led_count; i++) {
    uint32_t px_rgb = Adafruit_NeoPixel::getPixelColor(i);

    byte px_r = (px_rgb & 0x00FF0000) >> 16;
    byte px_g = (px_rgb & 0x0000FF00) >>  8;
    byte px_b = (px_rgb & 0x000000FF) >>  0;

    // fade out (divide by 2)
    px_r = px_r >> 1;
    px_g = px_g >> 1;
    px_b = px_b >> 1;

    Adafruit_NeoPixel::setPixelColor(i, px_r, px_g, px_b);
    yield();
  }

  if (random(3) == 0) {
    Adafruit_NeoPixel::setPixelColor(random(_led_count), _mode_color);
  }

  Adafruit_NeoPixel::show();

  _mode_delay = 100 + ((100 * (uint32_t)(SPEED_MAX - _speed)) / SPEED_MAX);
}


/*
   Blink several LEDs in random colors on, fading out.
*/
void WS2812FX::mode_twinkle_fade_random(void) {
  _mode_color = color_wheel(random(256));
  mode_twinkle_fade();
}


/*
   Lights all LEDs in the _color. Flashes single white pixels randomly.
   Inspired by www.tweaking4all.com/hardware/arduino/adruino-led-strip-effects/
*/
void WS2812FX::mode_flash_sparkle(void) {
  for (uint16_t i = 0; i < _led_count; i++) {
    Adafruit_NeoPixel::setPixelColor(i, _color);
    yield();
  }

  if (random(10) == 7) {
    Adafruit_NeoPixel::setPixelColor(random(_led_count), 255, 255, 255);
    _mode_delay = 20;
  } else {
    _mode_delay = 20 + ((200 * (uint32_t)(SPEED_MAX - _speed)) / SPEED_MAX);
  }

  Adafruit_NeoPixel::show();
}

/*
   _color running on white.
*/
void WS2812FX::mode_chase_white(void) {
  for (uint16_t i = 0; i < _led_count; i++) {
    Adafruit_NeoPixel::setPixelColor(i, 255, 255, 255);
    yield();
  }

  uint16_t n = _counter_mode_step;
  uint16_t m = (_counter_mode_step + 1) % _led_count;
  Adafruit_NeoPixel::setPixelColor(n, _color);
  Adafruit_NeoPixel::setPixelColor(m, _color);
  Adafruit_NeoPixel::show();

  _counter_mode_step = (_counter_mode_step + 1) % _led_count;
  _mode_delay = 10 + ((30 * (uint32_t)(SPEED_MAX - _speed)) / _led_count);
}


/*
   White running on _color.
*/
void WS2812FX::mode_chase_color(void) {
  for (uint16_t i = 0; i < _led_count; i++) {
    Adafruit_NeoPixel::setPixelColor(i, _color);
    yield();
  }

  uint16_t n = _counter_mode_step;
  uint16_t m = (_counter_mode_step + 1) % _led_count;
  Adafruit_NeoPixel::setPixelColor(n, 255, 255, 255);
  Adafruit_NeoPixel::setPixelColor(m, 255, 255, 255);
  Adafruit_NeoPixel::show();

  _counter_mode_step = (_counter_mode_step + 1) % _led_count;
  _mode_delay = 10 + ((30 * (uint32_t)(SPEED_MAX - _speed)) / _led_count);
}


/*
   White running followed by random color.
*/
void WS2812FX::mode_chase_random(void) {
  if (_counter_mode_step == 0) {
    Adafruit_NeoPixel::setPixelColor(_led_count - 1, color_wheel(_mode_color));
    _mode_color = get_random_wheel_index(_mode_color);
  }

  for (uint16_t i = 0; i < _counter_mode_step; i++) {
    Adafruit_NeoPixel::setPixelColor(i, color_wheel(_mode_color));
    yield();
  }

  uint16_t n = _counter_mode_step;
  uint16_t m = (_counter_mode_step + 1) % _led_count;
  Adafruit_NeoPixel::setPixelColor(n, 255, 255, 255);
  Adafruit_NeoPixel::setPixelColor(m, 255, 255, 255);

  Adafruit_NeoPixel::show();

  _counter_mode_step = (_counter_mode_step + 1) % _led_count;
  _mode_delay = 10 + ((30 * (uint32_t)(SPEED_MAX - _speed)) / _led_count);
}


/*
   White running on rainbow.
*/
void WS2812FX::mode_chase_rainbow(void) {
  for (uint16_t i = 0; i < _led_count; i++) {
    Adafruit_NeoPixel::setPixelColor(i, color_wheel(((i * 256 / _led_count) + (_counter_mode_call % 256)) % 256));
    yield();
  }

  uint16_t n = _counter_mode_step;
  uint16_t m = (_counter_mode_step + 1) % _led_count;
  Adafruit_NeoPixel::setPixelColor(n, 255, 255, 255);
  Adafruit_NeoPixel::setPixelColor(m, 255, 255, 255);
  Adafruit_NeoPixel::show();

  _counter_mode_step = (_counter_mode_step + 1) % _led_count;
  _mode_delay = 10 + ((30 * (uint32_t)(SPEED_MAX - _speed)) / _led_count);
}


/*
   White flashes running, followed by random color.
*/
void WS2812FX::mode_chase_flash_random(void) {
  const static uint8_t flash_count = 4;
  uint8_t flash_step = _counter_mode_call % ((flash_count * 2) + 1);

  for (uint16_t i = 0; i < _counter_mode_step; i++) {
    Adafruit_NeoPixel::setPixelColor(i, color_wheel(_mode_color));
    yield();
  }

  if (flash_step < (flash_count * 2)) {
    uint16_t n = _counter_mode_step;
    uint16_t m = (_counter_mode_step + 1) % _led_count;
    if (flash_step % 2 == 0) {
      Adafruit_NeoPixel::setPixelColor(n, 255, 255, 255);
      Adafruit_NeoPixel::setPixelColor(m, 255, 255, 255);
      _mode_delay = 20;
    } else {
      Adafruit_NeoPixel::setPixelColor(n, color_wheel(_mode_color));
      Adafruit_NeoPixel::setPixelColor(m, 0, 0, 0);
      _mode_delay = 30;
    }
  } else {
    _counter_mode_step = (_counter_mode_step + 1) % _led_count;
    _mode_delay = 1 + ((10 * (uint32_t)(SPEED_MAX - _speed)) / _led_count);

    if (_counter_mode_step == 0) {
      _mode_color = get_random_wheel_index(_mode_color);
    }
  }

  Adafruit_NeoPixel::show();
}

/*
   Black running on _color.
*/
void WS2812FX::mode_chase_blackout(void) {
  for (uint16_t i = 0; i < _led_count; i++) {
    Adafruit_NeoPixel::setPixelColor(i, _color);
    yield();
  }

  uint16_t n = _counter_mode_step;
  uint16_t m = (_counter_mode_step + 1) % _led_count;
  Adafruit_NeoPixel::setPixelColor(n, 0, 0, 0);
  Adafruit_NeoPixel::setPixelColor(m, 0, 0, 0);
  Adafruit_NeoPixel::show();

  _counter_mode_step = (_counter_mode_step + 1) % _led_count;
  _mode_delay = 10 + ((30 * (uint32_t)(SPEED_MAX - _speed)) / _led_count);
}


/*
   Black running on rainbow.
*/
void WS2812FX::mode_chase_blackout_rainbow(void) {
  for (uint16_t i = 0; i < _led_count; i++) {
    Adafruit_NeoPixel::setPixelColor(i, color_wheel(((i * 256 / _led_count) + (_counter_mode_call % 256)) % 256));
    yield();
  }

  uint16_t n = _counter_mode_step;
  uint16_t m = (_counter_mode_step + 1) % _led_count;
  Adafruit_NeoPixel::setPixelColor(n, 0, 0, 0);
  Adafruit_NeoPixel::setPixelColor(m, 0, 0, 0);
  Adafruit_NeoPixel::show();

  _counter_mode_step = (_counter_mode_step + 1) % _led_count;
  _mode_delay = 10 + ((30 * (uint32_t)(SPEED_MAX - _speed)) / _led_count);
}


/*
   Random color intruduced alternating from start and end of strip.
*/
void WS2812FX::mode_color_sweep_random(void) {
  if (_counter_mode_step == 0 || _counter_mode_step == _led_count) {
    _mode_color = get_random_wheel_index(_mode_color);
  }

  if (_counter_mode_step < _led_count) {
    Adafruit_NeoPixel::setPixelColor(_counter_mode_step, color_wheel(_mode_color));
  } else {
    Adafruit_NeoPixel::setPixelColor((_led_count * 2) - _counter_mode_step - 1, color_wheel(_mode_color));
  }
  Adafruit_NeoPixel::show();

  _counter_mode_step = (_counter_mode_step + 1) % (_led_count * 2);
  _mode_delay = 5 + ((50 * (uint32_t)(SPEED_MAX - _speed)) / _led_count);
}

/*
   Alternating red/blue pixels running.
*/
void WS2812FX::mode_running_red_blue(void) {
  for (uint16_t i = 0; i < _led_count; i++) {
    if ((i + _counter_mode_step) % 4 < 2) {
      Adafruit_NeoPixel::setPixelColor(i, 255, 0, 0);
    } else {
      Adafruit_NeoPixel::setPixelColor(i, 0, 0, 255);
    }
    yield();
  }
  Adafruit_NeoPixel::show();

  _counter_mode_step = (_counter_mode_step + 1) % 4;
  _mode_delay = 100 + ((100 * (uint32_t)(SPEED_MAX - _speed)) / _led_count);
}


/*
   Random colored pixels running.
*/
void WS2812FX::mode_running_random(void) {
  for (uint16_t i = _led_count - 1; i > 0; i--) {
    Adafruit_NeoPixel::setPixelColor(i, Adafruit_NeoPixel::getPixelColor(i - 1));
    yield();
  }

  if (_counter_mode_step == 0) {
    _mode_color = get_random_wheel_index(_mode_color);
    Adafruit_NeoPixel::setPixelColor(0, color_wheel(_mode_color));
  }

  Adafruit_NeoPixel::show();

  _counter_mode_step = (_counter_mode_step + 1) % 2;

  _mode_delay = 50 + ((50 * (uint32_t)(SPEED_MAX - _speed)) / _led_count);
}

/*
   Firework sparks.
*/
void WS2812FX::mode_fireworks(void) {
  uint32_t px_rgb = 0;
  byte px_r = 0;
  byte px_g = 0;
  byte px_b = 0;

  for (uint16_t i = 0; i < _led_count; i++) {
    px_rgb = Adafruit_NeoPixel::getPixelColor(i);

    px_r = (px_rgb & 0x00FF0000) >> 16;
    px_g = (px_rgb & 0x0000FF00) >>  8;
    px_b = (px_rgb & 0x000000FF) >>  0;

    // fade out (divide by 2)
    px_r = px_r >> 1;
    px_g = px_g >> 1;
    px_b = px_b >> 1;

    Adafruit_NeoPixel::setPixelColor(i, px_r, px_g, px_b);
    yield();
  }

  // first LED has only one neighbour
  px_r = (((Adafruit_NeoPixel::getPixelColor(1) & 0x00FF0000) >> 16) >> 1) + ((Adafruit_NeoPixel::getPixelColor(0) & 0x00FF0000) >> 16);
  px_g = (((Adafruit_NeoPixel::getPixelColor(1) & 0x0000FF00) >>  8) >> 1) + ((Adafruit_NeoPixel::getPixelColor(0) & 0x0000FF00) >>  8);
  px_b = (((Adafruit_NeoPixel::getPixelColor(1) & 0x000000FF) >>  0) >> 1) + ((Adafruit_NeoPixel::getPixelColor(0) & 0x000000FF) >>  0);
  Adafruit_NeoPixel::setPixelColor(0, px_r, px_g, px_b);

  for (uint16_t i = 1; i < _led_count - 1; i++) {
    yield();
    px_r = ((
              (((Adafruit_NeoPixel::getPixelColor(i - 1) & 0x00FF0000) >> 16) >> 1) +
              (((Adafruit_NeoPixel::getPixelColor(i + 1) & 0x00FF0000) >> 16) >> 0) ) >> 1) +
           (((Adafruit_NeoPixel::getPixelColor(i  ) & 0x00FF0000) >> 16) >> 0);

    px_g = ((
              (((Adafruit_NeoPixel::getPixelColor(i - 1) & 0x0000FF00) >> 8) >> 1) +
              (((Adafruit_NeoPixel::getPixelColor(i + 1) & 0x0000FF00) >> 8) >> 0) ) >> 1) +
           (((Adafruit_NeoPixel::getPixelColor(i  ) & 0x0000FF00) >> 8) >> 0);

    px_b = ((
              (((Adafruit_NeoPixel::getPixelColor(i - 1) & 0x000000FF) >> 0) >> 1) +
              (((Adafruit_NeoPixel::getPixelColor(i + 1) & 0x000000FF) >> 0) >> 0) ) >> 1) +
           (((Adafruit_NeoPixel::getPixelColor(i  ) & 0x000000FF) >> 0) >> 0);

    Adafruit_NeoPixel::setPixelColor(i, px_r, px_g, px_b);
  }

  // last LED has only one neighbour
  px_r = (((Adafruit_NeoPixel::getPixelColor(_led_count - 2) & 0x00FF0000) >> 16) >> 2) + ((Adafruit_NeoPixel::getPixelColor(_led_count - 1) & 0x00FF0000) >> 16);
  px_g = (((Adafruit_NeoPixel::getPixelColor(_led_count - 2) & 0x0000FF00) >>  8) >> 2) + ((Adafruit_NeoPixel::getPixelColor(_led_count - 1) & 0x0000FF00) >>  8);
  px_b = (((Adafruit_NeoPixel::getPixelColor(_led_count - 2) & 0x000000FF) >>  0) >> 2) + ((Adafruit_NeoPixel::getPixelColor(_led_count - 1) & 0x000000FF) >>  0);
  Adafruit_NeoPixel::setPixelColor(_led_count - 1, px_r, px_g, px_b);

  for (uint16_t i = 0; i < max(1, _led_count / 20); i++) {
    yield();
    if (random(10) == 0) {
      Adafruit_NeoPixel::setPixelColor(random(_led_count), _mode_color);
    }
  }

  Adafruit_NeoPixel::show();

  _mode_delay = 20 + ((20 * (uint32_t)(SPEED_MAX - _speed)) / _led_count);
}

/*
   Random colored firework sparks.
*/
void WS2812FX::mode_fireworks_random(void) {
  _mode_color = color_wheel(random(256));
  mode_fireworks();
}


/*
   Alternating red/green pixels running.
*/
void WS2812FX::mode_merry_christmas(void) {
  for (uint16_t i = 0; i < _led_count; i++) {
    if ((i + _counter_mode_step) % 4 < 2) {
      Adafruit_NeoPixel::setPixelColor(i, 255, 0, 0);
    } else {
      Adafruit_NeoPixel::setPixelColor(i, 0, 255, 0);
    }
    yield();
  }
  Adafruit_NeoPixel::show();

  _counter_mode_step = (_counter_mode_step + 1) % 4;
  _mode_delay = 100 + ((100 * (uint32_t)(SPEED_MAX - _speed)) / _led_count);
}

/*
   Random flickering.
*/
void WS2812FX::mode_fire_flicker(void) {
  mode_fire_flicker_int(6);
}

void WS2812FX::mode_fire_flicker_int(int rev_intensity)
{
  byte p_r = (_color & 0x00FF0000) >> 16;
  byte p_g = (_color & 0x0000FF00) >>  8;
  byte p_b = (_color & 0x000000FF) >>  0;
  byte flicker_val = max(p_r, max(p_g, p_b)) / rev_intensity;
  for (uint16_t i = 0; i < _led_count; i++)
  {
    int flicker = random(0, flicker_val);
    int r1 = p_r - flicker;
    int g1 = p_g - flicker;
    int b1 = p_b - flicker;
    if (g1 < 0) g1 = 0;
    if (r1 < 0) r1 = 0;
    if (b1 < 0) b1 = 0;
    Adafruit_NeoPixel::setPixelColor(i, r1, g1, b1);
    yield();
  }
  Adafruit_NeoPixel::show();
  _mode_delay = 10 + ((500 * (uint32_t)(SPEED_MAX - _speed)) / SPEED_MAX);
}

// CAL functions from here down

void WS2812FX::mode_auto(void) {
  _auto_mode = true;
  _auto_mode_change_time = 0;
  _mode_delay = 50;
  _autoTimeout = AUTO_TIMEOUT_MS + millis();
}

void WS2812FX::mode_off(void) {
  _auto_mode = false;
  strip_off();
}

// Geometric data
byte perimeterLEDs[][4] = {
  // Outside LEDs
  {  0,   1,  10,  11},
  { 26,  27,  28,  29},
  { 30,  31,  32,  33},
  { 38,  39,  40,  41},
  { 62,  63,  64,  65},
  { 86,  87,  88,  89},
  { 90,  91,  92,  93},
  { 98,  99, 100, 101},
  {122, 123, 124, 125},
  {146, 147, 148, 149},
  {150, 151, 152, 153},
  {162, 163, 164, 165},
  // Inside LEDs
  { 54,  55,  56,  57},
  {114, 115, 116, 117},
  {170, 171, 172, 173}
};

void WS2812FX::mode_perimeter(void) {

  if (_counter_mode_call > 14) {
    _counter_mode_call = 0;
  }
  uint32_t color = color_wheel(_counter_mode_step);

  for (int c = 0; c < 4; c++) {
    Adafruit_NeoPixel::setPixelColor(perimeterLEDs[_counter_mode_call][c], color);
  }
  Adafruit_NeoPixel::show();

  _counter_mode_step += 4;
  _counter_mode_step %= 256;

  _mode_delay = 5 + ((100 * (uint32_t)(SPEED_MAX - _speed)) / 56);
}

void WS2812FX::mode_perimeter_random(void) {

  if (_counter_mode_call > 14) {
    _counter_mode_call = 0;
  }
  uint32_t color = color_wheel(random(256));

  for (int c = 0; c < 4; c++) {
    Adafruit_NeoPixel::setPixelColor(perimeterLEDs[_counter_mode_call][c], color);
  }
  Adafruit_NeoPixel::show();

  _mode_delay = 5 + ((100 * (uint32_t)(SPEED_MAX - _speed)) / 56);
}


byte diag1LEDs[][4] = {
  {132, 133, 142, 143},
  {170, 171, 172, 173},
  { 18,  19,  20,  21}
};

byte diag2LEDs[][4] = {
  {12, 13, 22, 23},
  {54, 55, 56, 57},
  {74, 75, 76, 77}
};

byte diag3LEDs[][4] = {
  {138, 139, 140, 141},
  {114, 115, 116, 117},
  { 78,  79,  80,  81}
};

void WS2812FX::mode_diags(void) {
  Adafruit_NeoPixel::clear();

  if (_counter_mode_call > 2) {
    _counter_mode_call = 0;
  }
  uint32_t color = color_wheel(random(256));

  for (int r = 0; r < 3; r++) {
    for (int c = 0; c < 4; c++) {
      if (_counter_mode_call == 0) {
        Adafruit_NeoPixel::setPixelColor(diag1LEDs[r][c], color);
      } else if (_counter_mode_call == 1) {
        Adafruit_NeoPixel::setPixelColor(diag2LEDs[r][c], color);
      } else {
        Adafruit_NeoPixel::setPixelColor(diag3LEDs[r][c], color);
      }
    }
  }
  Adafruit_NeoPixel::show();

  _mode_delay = 5 + ((100 * (uint32_t)(SPEED_MAX - _speed)) / 36);
}

byte hLine0[][4] = {
  {84, 85, 94, 95}
};

byte hLine1[][4] = {
  {102, 103, 104, 105},
  { 60,  61,  70,  71}
};

byte hLine2[][4] = {
  {132, 133, 142, 143},
  {170, 171, 172, 173},
  { 18,  19,  20,  21}
};

byte hLine3[][4] = {
  {150, 151, 152, 153},
  {162, 163, 164, 165},
  {  0,   1,  10,  11},
  { 26,  27,  28,  29}
};

byte fSlash0[][4] = {
  {24, 25, 34, 35}
};

byte fSlash1[][4] = {
  {168, 169, 178, 179},
  { 48,  49,  58,  59}
};

byte fSlash2[][4] = {
  {138, 139, 140, 141},
  {114, 115, 116, 117},
  { 78,  79,  80,  81}
};

byte fSlash3[][4] = {
  {146, 147, 148, 149},
  {122, 123, 124, 125},
  { 98,  99, 100, 101},
  { 90,  91,  92,  93}
};

byte bSlash0[][4] = {
  {144, 145, 154, 155},
};

byte bSlash1[][4] = {
  {156, 157, 166, 167},
  {120, 121, 130, 131},
};

byte bSlash2[][4] = {
  {12, 13, 22, 23},
  {54, 55, 56, 57},
  {74, 75, 76, 77}
};

byte bSlash3[][4] = {
  {30, 31, 32, 33},
  {38, 39, 40, 41},
  {62, 63, 64, 65},
  {86, 87, 88, 89},
};

void WS2812FX::mode_inward(void) {
  Adafruit_NeoPixel::clear();

  if (_counter_mode_call > 12) {
    _counter_mode_call = 0;
  }
  uint32_t color = color_wheel(random(256));

  switch (_counter_mode_call) {
    case 0:
      for (int c = 0; c < 4; c++) {
        Adafruit_NeoPixel::setPixelColor(bSlash0[0][c], color);
      }
      break;
    case 1:
      for (int r = 0; r < 2; r++) {
        for (int c = 0; c < 4; c++) {
          Adafruit_NeoPixel::setPixelColor(bSlash1[r][c], color);
        }
      }
      break;
    case 2:
      for (int r = 0; r < 3; r++) {
        for (int c = 0; c < 4; c++) {
          Adafruit_NeoPixel::setPixelColor(bSlash2[r][c], color);
        }
      }
      break;
    case 3:
      for (int r = 0; r < 4; r++) {
        for (int c = 0; c < 4; c++) {
          Adafruit_NeoPixel::setPixelColor(bSlash3[r][c], color);
        }
      }
      break;
    case 4:
      for (int c = 0; c < 4; c++) {
        Adafruit_NeoPixel::setPixelColor(fSlash0[0][c], color);
      }
      break;
    case 5:
      for (int r = 0; r < 2; r++) {
        for (int c = 0; c < 4; c++) {
          Adafruit_NeoPixel::setPixelColor(fSlash1[r][c], color);
        }
      }
      break;
    case 6:
      for (int r = 0; r < 3; r++) {
        for (int c = 0; c < 4; c++) {
          Adafruit_NeoPixel::setPixelColor(fSlash2[r][c], color);
        }
      }
      break;
    case 7:
      for (int r = 0; r < 4; r++) {
        for (int c = 0; c < 4; c++) {
          Adafruit_NeoPixel::setPixelColor(fSlash3[r][c], color);
        }
      }
      break;

    case 8:
      for (int c = 0; c < 4; c++) {
        Adafruit_NeoPixel::setPixelColor(hLine0[0][c], color);
      }
      break;
    case 9:
      for (int r = 0; r < 2; r++) {
        for (int c = 0; c < 4; c++) {
          Adafruit_NeoPixel::setPixelColor(hLine1[r][c], color);
        }
      }
      break;
    case 10:
      for (int r = 0; r < 3; r++) {
        for (int c = 0; c < 4; c++) {
          Adafruit_NeoPixel::setPixelColor(hLine2[r][c], color);
        }
      }
      break;
    case 11:
      for (int r = 0; r < 4; r++) {
        for (int c = 0; c < 4; c++) {
          Adafruit_NeoPixel::setPixelColor(hLine3[r][c], color);
        }
      }
      break;
  }
  Adafruit_NeoPixel::show();

  _mode_delay = 5 + ((100 * (uint32_t)(SPEED_MAX - _speed)) / 36);
}

// Support routine that fills a specified triangle with a color
// Triangles are numbered 0 to 14
void WS2812FX::drawATriangle(int num, uint32_t color) {
  int startPixel = num * 12;
  int endPixel   = (num + 1) * 12;

  for (int i = startPixel; i < endPixel; i++) {
    Adafruit_NeoPixel::setPixelColor(i, color);
  }
  Adafruit_NeoPixel::show();
}

void WS2812FX::mode_triangles(void) {

  // Pick a triangle to color
  int triNum = random(15);

  // Either paint a color or paint it black
  int pick = random(2);

  if (pick == 0) {
    drawATriangle(triNum, 0);
  } else {
    drawATriangle(triNum, color_wheel(_counter_mode_step));
  }
  _counter_mode_step += 2;
  _counter_mode_step %= 256;

  _mode_delay = 5 + ((100 * (uint32_t)(SPEED_MAX - _speed)) / 36);
}
