/*
  TriLight - ESP32 Controlled Triangle Light

  This program creates a web server for controlling a 3D printed light containing 180
  NeoPixels for various lighting effects. A web UI is created for controlling the effects.
  The UI is based upon the web interface example program from the WS2812FX library by Harm Aldick
  (www.aldick.org) which allows the mode, color, speed and brightness to be controlled over the web.

  This code uses the ArduinoOTA (over the air) update mechanism to update the TriLight
  code instead of having to connect it to the development computer via USB. To use OTA update
  define OTA and recompile and upload the code.

  Code was compiled with version 1.8.16 of the Arduino IDE on macOS and a custom version
  of the WS2812FX library embedded in this code.

  Porting of the example program and all new features by: Craig A. Lindley
  Last Update: 12/23/2021
*/

#include <WiFi.h>
#include <WebServer.h>

#define OTA // define this to use OTA code updates
#ifdef OTA
#include <ArduinoOTA.h>
#endif

#include "WS2812FX.h"
#include "index.html.h"
#include "main.js.h"

#define LED_PIN              4
#define LED_COUNT          180

#define WIFI_SSID       "CraigNet"
#define WIFI_PSWD    "craigandheather"
#define WIFI_HOSTNAME  "TriLight"
#define WIFI_TIMEOUT    60000             // Checks WiFi every minute. Reboot if connection down
#define HTTP_PORT          80

#define DEFAULT_COLOR      0xFF5900
#define DEFAULT_BRIGHTNESS 40
#define DEFAULT_SPEED      100
#define DEFAULT_MODE       FX_MODE_AUTO

#define BRIGHTNESS_STEP    10          // in/decrease brightness by this amount per click
#define SPEED_STEP         15          // in/decrease brightness by this amount per click

unsigned long last_wifi_check_time = 0;

String modes = "";

/************************************************************************/
/*                     Object Instantiations                            */
/************************************************************************/

// Instantiate the NeoPixel driver
WS2812FX ws2812fx = WS2812FX(LED_COUNT, LED_PIN, NEO_GRB + NEO_KHZ800);

// Instantiate a web server
WebServer server(HTTP_PORT);

/************************************************************************/
/*                        Misc Webserver Functions                      */
/************************************************************************/
// Check for Internet connectivity by pinging google DNS
// Returns true if Internet is accessible
boolean checkInternetConnection() {
  IPAddress serverIP;

  int result = WiFi.hostByName("8.8.8.8", serverIP);
  return (result == 1);
}

void srv_handle_not_found() {
  server.send(404, "text/plain", "File Not Found");
}

void srv_handle_index_html() {
  server.send_P(200, "text/html", index_html);
}

void srv_handle_main_js() {
  server.send_P(200, "application/javascript", main_js);
}

void srv_handle_modes() {
  server.send(200, "text/plain", modes);
}

void srv_handle_set() {

  for (uint8_t i = 0; i < server.args(); i++) {
    // Color change ?
    if (server.argName(i) == "c") {
      String s = server.arg(i);
      uint32_t tmp = (uint32_t) strtol(s.c_str(), NULL, 16);
      if (tmp >= 0x000000 && tmp <= 0xFFFFFF) {
        ws2812fx.setColor(tmp);
      }
    }

    // Mode change ?
    if (server.argName(i) == "m") {
      String s = server.arg(i);
      uint8_t tmp = (uint8_t) strtol(s.c_str(), NULL, 10);
      ws2812fx.setMode(tmp % ws2812fx.getModeCount());
      Serial.println(ws2812fx.getModeName(tmp));  // CAL
    }

    // Brightness change ?
    if (server.argName(i) == "b") {
      if (server.arg(i)[0] == '-') {
        ws2812fx.decreaseBrightness(BRIGHTNESS_STEP);
      } else {
        ws2812fx.increaseBrightness(BRIGHTNESS_STEP);
      }
    }

    // Speed change ?
    if (server.argName(i) == "s") {
      if (server.arg(i)[0] == '-') {
        ws2812fx.decreaseSpeed(SPEED_STEP);
      } else {
        ws2812fx.increaseSpeed(SPEED_STEP);
      }
    }
  }
  server.send(200, "text/plain", "OK");
}

// Build <li> string for all modes.
void modes_setup() {
  modes = "";
  for (uint8_t i = 0; i < ws2812fx.getModeCount(); i++) {
    modes += "<li><a href='#' class='m' id='";
    modes += i;
    modes += "'>";
    modes += ws2812fx.getModeName(i);
    modes += "</a></li>";
    yield();
  }
}

/************************************************************************/
/*                            Program Setup                             */
/************************************************************************/

void setup() {

  Serial.begin(115200);
  delay(1000);
  Serial.println("\n\nStarting...");

  // Seed random number generator
  randomSeed(analogRead(0));

#ifdef OTA
  ArduinoOTA.setHostname("TriLight");

  ArduinoOTA.onStart([]() {
    Serial.println("Start");
  });

  ArduinoOTA.onEnd([]() {
    Serial.println("End");
  });

  ArduinoOTA.onProgress([](unsigned int progress, unsigned int total) {
    Serial.printf("Progress: %u%%\r", (progress / (total / 100)));
  });

  ArduinoOTA.onError([](ota_error_t error) {
    Serial.printf("Error[%u]: ", error);
    if (error == OTA_AUTH_ERROR) Serial.println("Auth Failed");
    else if (error == OTA_BEGIN_ERROR) Serial.println("Begin Failed");
    else if (error == OTA_CONNECT_ERROR) Serial.println("Connect Failed");
    else if (error == OTA_RECEIVE_ERROR) Serial.println("Receive Failed");
    else if (error == OTA_END_ERROR) Serial.println("End Failed");
  });
#endif

  // Setup WS2812FX library
  Serial.println("WS2812FX setup");
  ws2812fx.init();
  ws2812fx.setMode(DEFAULT_MODE);
  ws2812fx.setColor(DEFAULT_COLOR);
  ws2812fx.setSpeed(DEFAULT_SPEED);
  ws2812fx.setBrightness(DEFAULT_BRIGHTNESS);
  ws2812fx.start();

  // Bring up WiFi on local network
  WiFi.disconnect();
  WiFi.mode(WIFI_STA);
  WiFi.hostname(WIFI_HOSTNAME);
  WiFi.begin(WIFI_SSID, WIFI_PSWD);
  int i = 0;
  while ((WiFi.status() != WL_CONNECTED) && (i++ < 100)) {
    Serial.print(".");
    delay(500);
  }

  // Did we connect ?
  if (WiFi.status() == WL_CONNECTED) {
    Serial.println("\n\nWiFi connected");
  } else {
    Serial.println("\n\nWiFi did not connect");
    delay(3000);
    ESP.restart();
  }

  // Install web handlers
  server.on("/", srv_handle_index_html);
  server.on("/main.js", srv_handle_main_js);
  server.on("/modes", srv_handle_modes);
  server.on("/set", srv_handle_set);
  server.onNotFound(srv_handle_not_found);

  delay(100);

  // Start the web server
  server.begin();

#ifdef OTA
  // Start over the air updates
  ArduinoOTA.begin();
#endif

  Serial.println("Server started - Program ready!");

  // Gather all of the available mode names into a unordered list
  // for sending to the browser
  modes.reserve(2500);
  modes_setup();
}

/************************************************************************/
/*                             Program Loop                             */
/************************************************************************/

void loop() {

  server.handleClient();
  ws2812fx.service();

#ifdef OTA
  ArduinoOTA.handle();
#endif

  unsigned long now = millis();

  yield();

  if (now - last_wifi_check_time > WIFI_TIMEOUT) {
    Serial.print("Checking WiFi... ");
    if (WiFi.status() != WL_CONNECTED) {
      Serial.println("WiFi connection lost. Restarting...\n\n\n");
      delay(3000);
      ESP.restart();
    } else {
      Serial.println("WiFi OK");
    }
    last_wifi_check_time = now;

    // Now check for Internet connectivity
    if (checkInternetConnection() == false) {
      Serial.println("WiFi connection lost. Restarting...\n\n\n");
      delay(3000);
      ESP.restart();
    }
  }
}
