/*
    NTP.h - Network Time Protocol Functions

    Portions of this code were extracted from the
    Time library examples by Michael Margolis and
    Paul Stoffregen. Other portions from the NTPClient
    example program from the Arduino ESP8266 package.

    Concept, Design and Implementation by: Craig A. Lindley
    Last Update: 12/31/2024
*/

#ifndef NTP_H
#define NTP_H

#include <WiFiUDP.h>

// Define the time between sync events
#define SYNC_INTERVAL_HOURS 4
#define SYNC_INTERVAL_MINUTES (SYNC_INTERVAL_HOURS * 60L)
#define SYNC_INTERVAL_SECONDS (SYNC_INTERVAL_MINUTES * 60L)

#define LOCALPORT 2390      // Local port to listen for UDP packets
#define NTP_PACKET_SIZE 48  // NTP time stamp is in the first 48 bytes of the message

// Buffer to hold incoming and outgoing packets
byte packetBuffer[NTP_PACKET_SIZE + 10];

// Don't hardwire the IP address or we won't get the benefits of the time server pool.
IPAddress timeServerIP;
const char *ntpServerName = "pool.ntp.org";

// NTP Time Provider Code
time_t getNTPTime() {

  int attempts = 10;

  // Try multiple times to return the NTP time
  while (attempts--) {
    // Create UDP instance for sending and receiving packets over UDP
    WiFiUDP udp;

    // Get a server from the pool
    WiFi.hostByName(ntpServerName, timeServerIP);
    Serial.print("Time server IP address: ");
    Serial.println(timeServerIP);

    // Set UDP local port
    udp.begin(LOCALPORT);

    // Set all bytes in the buffer to 0
    memset(packetBuffer, 0, sizeof(packetBuffer));

    // Initialize values needed to form NTP request
    packetBuffer[0] = 0xE3;
    packetBuffer[2] = 0x06;
    packetBuffer[3] = 0xEC;
    packetBuffer[12] = 0x31;
    packetBuffer[13] = 0x4E;
    packetBuffer[14] = 0x31;
    packetBuffer[15] = 0x34;

    // All necessary NTP fields have been given values, now
    // you can send a packet requesting a timestamp:
    udp.beginPacket(timeServerIP, 123);  // NTP requests to port 123
    udp.write(packetBuffer, NTP_PACKET_SIZE);
    udp.endPacket();

    delay(100);

    uint32_t beginWait = millis();
    while (millis() - beginWait < 1500) {
      int size = udp.parsePacket();
      if (size >= NTP_PACKET_SIZE) {
        Serial.println("Received NTP Response");
        udp.read(packetBuffer, NTP_PACKET_SIZE);  // Read packet into the buffer
        unsigned long secsSince1900;

        // Convert four bytes starting at location 40 to a long integer
        secsSince1900 = (unsigned long)packetBuffer[40] << 24;
        secsSince1900 |= (unsigned long)packetBuffer[41] << 16;
        secsSince1900 |= (unsigned long)packetBuffer[42] << 8;
        secsSince1900 |= (unsigned long)packetBuffer[43];

        Serial.println("Got the time");
        udp.stop();
        return secsSince1900 - 2208988800UL;
      }
    }
    udp.stop();
    Serial.println("Retrying NTP request");
    delay(60000);
  }
  // Should rarely if ever get here
  Serial.println("No NTP Response");

  // Restart ESP32 if this occurs
  esp_restart();

  // Will never get here
  return 0;
}

// Assign NTP as the time sync provider
void initNTP() {

  // Set the time provider to NTP
  setSyncProvider(getNTPTime);

  // Set the interval between NTP calls
  setSyncInterval(SYNC_INTERVAL_SECONDS);
}

// Check for Internet connectivity
boolean checkInternetConnection() {
  IPAddress timeServerIP;

  int result = WiFi.hostByName(ntpServerName, timeServerIP);
  return (result == 1);
}

#endif
