/*
 * Clock Calendar Program
 *
 * Utilizing an ESP32S3 and a Elecrow 7" LCD Panel Display
 * Resolution 800x480
 *
 * Compile with Board: ESP32S3 Dev Module
 * PSRAM: OPI PSRAM
 * Upload Speed: 460800
 *
 * Change the network credentials in the code below
 *
 * Concept, Design and Implementation by: Craig A. Lindley
 * Last Update: 12/30/2024
 */

#include <WiFi.h>
#include <LovyanGFX.hpp>

#include "Time.h"
#include "Timezone.h"
#include "NTP.h"
#include "LGFX_Elecrow_ESP32_Display_WZ8048C070.h"

/************************************************************************/
/*                         Timezone properties                          */
/************************************************************************/

// Change the following for your timezone

// This clock runs in the Mountain Time Zone
#define DST_TIMEZONE_OFFSET -6  // Day Light Saving Time offset (-6 is mountain time)
#define ST_TIMEZONE_OFFSET -7   // Standard Time offset (-7 is mountain time)

// TimeZone and Daylight Savings Time Rules
// Define daylight savings time rules for the Mountain Time Zone
TimeChangeRule myDST = { "MDT", Second, Sun, Mar, 2, DST_TIMEZONE_OFFSET * 60 };
TimeChangeRule mySTD = { "MST", First, Sun, Nov, 2, ST_TIMEZONE_OFFSET * 60 };
Timezone myTZ(myDST, mySTD);

/************************************************************************/
/*                                Drivers                               */
/************************************************************************/

// Instantiate the LCD driver
static LGFX lcd;

/************************************************************************/
/*                        Constants and Variables                       */
/************************************************************************/

// These are my network credentials, change them for yours
#define WIFI_SSID "CraigNet"
#define WIFI_PSWD "craigandheather"

// Define display 16bit colors
#define BLACK 0x0000
#define WHITE 0xffff
#define RED 0xf800
#define GREEN 0x07e0
#define BLUE 0x001f
#define YELLOW 0xffe0
#define CYAN 0x07ff
#define MAGENTA 0xf81f
#define BROWN oxA145

// Clock rounded rect parameters
#define CLOCK_X_CENTER 200
#define CLOCK_Y_CENTER 240
#define CLOCK_OUTER_WIDTH 340
#define CLOCK_OUTER_HEIGHT 340
#define CLOCK_MIDDLE_WIDTH 300
#define CLOCK_MIDDLE_HEIGHT 300
#define CLOCK_INNER_WIDTH 240
#define CLOCK_INNER_HEIGHT 240
#define CLOCK_RR2N 4

// Clock hands parameters
#define CLOCK_ERASE_RADIUS 100
#define CLOCK_SEC_HAND_LONG_LEN 98
#define CLOCK_SEC_HAND_SHORT_LEN 35
#define CLOCK_SEC_HAND_CIRCLE_RADIUS 4
#define CLOCK_MIN_HAND_LONG_LEN 95
#define CLOCK_MIN_HAND_SHORT_LEN 35
#define CLOCK_MIN_HAND_WIDTH 3
#define CLOCK_HOUR_HAND_LONG_LEN 78
#define CLOCK_HOUR_HAND_SHORT_LEN 35
#define CLOCK_HOUR_HAND_WIDTH 3
#define CLOCK_CIRCLE_RADIUS 6

// Calendar parameters
#define CALENDAR_X_OFFSET 400
#define CALENDAR_BORDER_WIDTH 50
#define CALENDAR_LINE_HEIGHT 35
#define CALENDAR_CELL_WIDTH 32
#define CALENDAR_CELL_SPACE 15

// Text offsets in calendar cell
#define TEXT_V_OFFSET 4
#define TEXT_H_1_CHAR_OFFSET 12
#define TEXT_H_2_CHAR_OFFSET 4

#define BOTTOM_LABEL "Craig A. Lindley"

int prevSecond;
int prevMinute;
int prevDay;

uint32_t futureTime;

// Finite State Machine (FSM) states
enum STATES { STATE_INIT,
              STATE_RUN,
              STATE_TURN_OFF,
              STATE_WAIT
};

enum STATES state;

/************************************************************************/
/*                              Misc Functions                          */
/************************************************************************/

// Control the backlight
// true is on; false if off
void backlight(bool blState) {
  if (blState) {
    // Turn backlight on
    lcd.setBrightness(255);
  } else {
    // Turn backlight off
    lcd.setBrightness(0);
  }
}

/************************************************************************/
/*                             Clock Functions                          */
/************************************************************************/

// Function to draw both minute and hour hands
void drawMinHourHand(int angle, int handLongLength, int handShortLength, int handWidth) {

  // Calculate positions of the two circles
  float xpos = round(CLOCK_X_CENTER + sin(radians(angle)) * handLongLength);
  float ypos = round(CLOCK_Y_CENTER - cos(radians(angle)) * handLongLength);
  float xpos2 = round(CLOCK_X_CENTER + sin(radians(angle)) * handShortLength);
  float ypos2 = round(CLOCK_Y_CENTER - cos(radians(angle)) * handShortLength);

  float tri_xoff = round(sin(radians(angle + 90)) * handWidth);
  float tri_yoff = round(-cos(radians(angle + 90)) * handWidth);

  lcd.drawLine(CLOCK_X_CENTER, CLOCK_Y_CENTER, xpos2, ypos2, WHITE);  // Draw line from one circle to the center
  lcd.fillCircle(xpos, ypos, handWidth, WHITE);
  lcd.fillCircle(xpos2, ypos2, handWidth, WHITE);

  // Two filled triangles are used to draw a rotated rectangle between two circles
  lcd.fillTriangle(xpos + tri_xoff, ypos + tri_yoff,
                   xpos - tri_xoff, ypos - tri_yoff,
                   xpos2 + tri_xoff, ypos2 + tri_yoff, WHITE);

  lcd.fillTriangle(xpos2 + tri_xoff, ypos2 + tri_yoff,
                   xpos2 - tri_xoff, ypos2 - tri_yoff,
                   xpos - tri_xoff, ypos - tri_yoff, WHITE);
}

// Draw second hand
void drawSecondHand(int angle, int handLongLength, int handShortLength) {

  // Calculate starting and ending position of the second hand
  float xpos = round(CLOCK_X_CENTER + sin(radians(angle)) * handLongLength);
  float ypos = round(CLOCK_Y_CENTER - cos(radians(angle)) * handLongLength);
  float xpos2 = round(CLOCK_X_CENTER + sin(radians(angle + 180)) * handShortLength);
  float ypos2 = round(CLOCK_Y_CENTER - cos(radians(angle + 180)) * handShortLength);

  lcd.drawLine(xpos, ypos, xpos2, ypos2, WHITE);
  lcd.fillCircle(xpos2, ypos2, CLOCK_SEC_HAND_CIRCLE_RADIUS, BLACK);
  lcd.drawCircle(xpos2, ypos2, CLOCK_SEC_HAND_CIRCLE_RADIUS, WHITE);
}

// Calculate a point X, Y on a rounded rect
void calculatePointOnRoundedRect(int angle, int width, int height, int *pX, int *pY) {

  double rads = radians(angle);
  double aCos = cos(rads);
  double aSin = sin(rads);

  double term1 = pow(aCos / (height / 2.0), CLOCK_RR2N);
  double term2 = pow(aSin / (width / 2.0), CLOCK_RR2N);

  double p = 1.0 / pow(term1 + term2, 1.0 / CLOCK_RR2N);

  *pX = CLOCK_X_CENTER + p * aCos;
  *pY = CLOCK_Y_CENTER + p * aSin;
}

// Draw the clock's face which is static
// Clock hands are handled separately
void drawClockFace() {
  int x, y, x1, y1;

  // Draw minute tics
  for (int angle = 0; angle < 360; angle += 6) {
    calculatePointOnRoundedRect(angle, CLOCK_OUTER_WIDTH, CLOCK_OUTER_HEIGHT, &x, &y);
    calculatePointOnRoundedRect(angle, CLOCK_MIDDLE_WIDTH, CLOCK_MIDDLE_HEIGHT, &x1, &y1);

    lcd.drawLine(x, y, x1, y1, GREEN);
  }

  // Draw 5 minute tics
  for (int angle = 0; angle < 360; angle += 30) {
    calculatePointOnRoundedRect(angle, CLOCK_OUTER_WIDTH, CLOCK_OUTER_HEIGHT, &x, &y);
    calculatePointOnRoundedRect(angle, CLOCK_INNER_WIDTH, CLOCK_INNER_HEIGHT, &x1, &y1);

    lcd.drawLine(x, y, x1, y1, WHITE);
  }
  // Label the four reference hours
  // Label 12 o'clock
  x = CLOCK_X_CENTER - 14;
  y = CLOCK_Y_CENTER - 117;
  lcd.setCursor(x, y);
  lcd.print("12");

  // Label 6 o'clock
  x = CLOCK_X_CENTER - 6;
  y = CLOCK_Y_CENTER + 100;
  lcd.setCursor(x, y);
  lcd.print("6");

  // Label 9 o'clock
  x = CLOCK_X_CENTER - 116;
  y = CLOCK_Y_CENTER - 8;
  lcd.setCursor(x, y);
  lcd.print("9");

  // Label 3 o'clock
  x = CLOCK_X_CENTER + 106;
  y = CLOCK_Y_CENTER - 8;
  lcd.setCursor(x, y);
  lcd.print("3");
}

// Update time display
void updateTime(int hours, int minutes, int seconds) {

  // First erase clock hands
  lcd.fillCircle(CLOCK_X_CENTER, CLOCK_Y_CENTER, CLOCK_ERASE_RADIUS, BLACK);

  // Draw second hand
  drawSecondHand(seconds * 6, CLOCK_SEC_HAND_LONG_LEN, CLOCK_SEC_HAND_SHORT_LEN);

  // Draw minute hand
  drawMinHourHand(minutes * 6, CLOCK_MIN_HAND_LONG_LEN, CLOCK_MIN_HAND_SHORT_LEN, CLOCK_MIN_HAND_WIDTH);

  // Draw hour hand
  // Move hour hand slowly towards next hour
  drawMinHourHand(hours * 30 + (minutes / 2), CLOCK_HOUR_HAND_LONG_LEN, CLOCK_HOUR_HAND_SHORT_LEN, CLOCK_HOUR_HAND_WIDTH);

  // Clear clock center circle
  lcd.fillCircle(CLOCK_X_CENTER, CLOCK_Y_CENTER, CLOCK_CIRCLE_RADIUS, BLACK);

  // Draw white outline
  lcd.drawCircle(CLOCK_X_CENTER, CLOCK_Y_CENTER, CLOCK_CIRCLE_RADIUS, WHITE);
}

// Display time in digital form
// ex. 12:36 AM
void displayDigitalTime(int hours, int minutes, bool isAM) {
  char tBuffer[20];

  // Clear digital time display area
  lcd.fillRect(0, lcd.height() - 55, lcd.width() / 2, lcd.height() - 55, BLACK);

  // Format the time into the buffer
  sprintf(tBuffer, "%d : %02d %s", hours, minutes, isAM ? "AM" : "PM");
  lcd.setTextColor(WHITE);
  int len = lcd.textWidth(tBuffer);
  int xOffset = ((lcd.width() / 2) - len) / 2;
  int yOffset = lcd.height() - 30;
  lcd.setCursor(xOffset, yOffset);
  lcd.print(tBuffer);
}

/************************************************************************/
/*                          Calendar Functions                          */
/************************************************************************/

// Indices are 1 .. 12
const char *MONTHNAMES[] = {
  "NA",
  "JANUARY", "FEBRUARY", "MARCH", "APRIL",
  "MAY", "JUNE", "JULY", "AUGUST",
  "SEPTEMBER", "OCTOBER", "NOVEMBER", "DECEMBER"
};

// Array of days in each month
// Indices are 1 .. 12
int MONTHDAYS[] = {
  0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31
};

// Determine if the specified year is a leap year
// Returns true is so and false otherwise
bool isLeapYear(int year) {
  return ((year % 4 == 0 && year % 100 != 0) || year % 400 == 0);
}

// Get days in the specified month and year
// Year is current year i.e. 2024 and month is 1 .. 12
int getMonthDays(int year, int month) {

  int dim = MONTHDAYS[month];
  if (isLeapYear(year) && (month == 2)) {
    return dim + 1;
  } else {
    return dim;
  }
}

// Get days in previous month and year
int getPrevMonthDays(int year, int month) {
  if (month == 1) {
    return getMonthDays(year - 1, 12);
  } else {
    return getMonthDays(year, month - 1);
  }
}

// Calculate X, Y location of specific calendar col/row
void calculateColRowLocation(int col, int row, int *pX, int *pY) {

  *pY = row * CALENDAR_LINE_HEIGHT;
  *pX = CALENDAR_X_OFFSET + CALENDAR_BORDER_WIDTH + col * (CALENDAR_CELL_WIDTH + CALENDAR_CELL_SPACE);
}

// Draw one or two char text string at col/row
void drawTextAtColRow(int col, int row, const char *text) {

  int x, y;

  // Calculate location of col/row
  calculateColRowLocation(col, row, &x, &y);

  int len = strlen(text);
  if (len == 1) {
    // 1 char
    x += TEXT_H_1_CHAR_OFFSET;
  } else {
    // 2 chars
    x += TEXT_H_2_CHAR_OFFSET;
  }
  lcd.setCursor(x, y);
  lcd.print(text);
}

// Draw 1 or 2 digit number at row/column
void drawNumberAtColRow(int col, int row, int num) {
  String s = String(num);
  drawTextAtColRow(col, row, s.c_str());
}

// Draw calendar header for given month
void drawCalendarHeaderForMonth(int year, int month) {

  // Set default text color
  lcd.setTextColor(WHITE);

  // Clear calendar side of display
  lcd.fillRect(CALENDAR_X_OFFSET, 60, lcd.width() - 1, lcd.height() - 90, BLACK);

  // Print month and year large
  lcd.setTextSize(2, 2);

  char yBuffer[20];
  sprintf(yBuffer, "%d", year);

  int len = lcd.textWidth(yBuffer);
  int xOffset = ((CALENDAR_X_OFFSET - len) / 2) + CALENDAR_X_OFFSET;
  int yOffset = CALENDAR_LINE_HEIGHT;
  lcd.setCursor(xOffset, yOffset);
  lcd.print(yBuffer);

  // Get month name
  const char *mN = MONTHNAMES[month];
  len = lcd.textWidth(mN);
  xOffset = ((CALENDAR_X_OFFSET - len) / 2) + CALENDAR_X_OFFSET;
  yOffset = CALENDAR_LINE_HEIGHT * 3;
  lcd.setCursor(xOffset, yOffset);
  lcd.print(mN);

  // Put normal text size back
  lcd.setTextSize(1, 1);

  // Draw a dividing line
  int x1 = CALENDAR_X_OFFSET + CALENDAR_BORDER_WIDTH;
  int y1 = CALENDAR_LINE_HEIGHT * 4 + 15;
  int x2 = lcd.width() - CALENDAR_BORDER_WIDTH - 1;
  int y2 = y1;
  lcd.drawLine(x1, y1, x2, y2, GREEN);

  // Draw days of the week
  drawTextAtColRow(0, 5, "S");
  drawTextAtColRow(1, 5, "M");
  drawTextAtColRow(2, 5, "T");
  drawTextAtColRow(3, 5, "W");
  drawTextAtColRow(4, 5, "Th");
  drawTextAtColRow(5, 5, "F");
  drawTextAtColRow(6, 5, "S");
}

// Draw the Calendar
void drawCalendar(uint32_t localTimeSeconds) {

  int curDay = day(localTimeSeconds);
  int curMon = month(localTimeSeconds);
  int curYear = year(localTimeSeconds);
  int monDays = getMonthDays(curYear, curMon);
  int prevMonDays = getPrevMonthDays(curYear, curMon);

  // Calculate the day of the week for the first day of the month
  // NOTE: this uses system time not the time library for
  // calculations.
  struct tm tm = { 0 };
  tm.tm_mday = 1;
  tm.tm_mon = curMon - 1;                    // Months are 0-based in tm structure
  tm.tm_year = curYear - 1900;               // Years are years since 1900 in tm structure
  int firstDayOfMon = weekday(mktime(&tm));  // Get the 1st day of the month

  // Draw calendar header for specified year and month
  drawCalendarHeaderForMonth(curYear, curMon);

  // Initialize variables for calendar
  bool done = false;
  int dayCount = 1;
  int row = 6;
  int col = 0;
  int state = 0;

  // Run a FSM
  while (!done) {
    switch (state) {
      case 0:
        // Check if 1st day of month is Sun
        if (firstDayOfMon == 1) {
          // Yes calendar starts on Sun
          state = 1;
        } else {
          // Calendar starts on some other day
          state = 2;
        }
        break;

      case 1:
        if (curDay == dayCount) {
          // Set highlight color
          lcd.setTextColor(RED);
        } else {
          // Select normal color
          lcd.setTextColor(WHITE);
        }
        drawNumberAtColRow(col, row, dayCount);
        col++;
        dayCount++;

        // Are we done with this month's days ?
        if (dayCount > monDays) {
          state = 3;
        } else if (col == 7) {
          col = 0;
          row++;
        }
        break;

      case 2:
        // Show days from previous month
        lcd.setTextColor(GREEN);
        dayCount = prevMonDays - firstDayOfMon + 2;
        for (int i = 0; i < firstDayOfMon - 1; i++) {
          drawNumberAtColRow(col, row, dayCount);
          col++;
          dayCount++;
        }
        dayCount = 1;
        col = firstDayOfMon - 1;
        state = 1;
        break;

      case 3:
        // Show days starting next month
        lcd.setTextColor(GREEN);
        dayCount = 1;
        while (col != 7) {
          drawNumberAtColRow(col, row, dayCount);
          col++;
          dayCount++;
        }
        done = true;
        break;
    }
  }
}

/************************************************************************/
/*                             Program Setup                            */
/************************************************************************/

void setup() {

  Serial.begin(115200);
  delay(10000);

  // Wi-Fi connection
  WiFi.mode(WIFI_OFF);

  int attempts = 0;

  WiFi.begin(WIFI_SSID, WIFI_PSWD);
  while (WiFi.status() != WL_CONNECTED) {
    Serial.print(".");
    delay(1000);
    attempts++;
    if (attempts > 60) {
      esp_restart();
    }
  }
  Serial.println("\nWiFi connected");

  // Initialize NTP
  initNTP();

  // Initialize LCD display
  lcd.init();
  lcd.setRotation(0);
  lcd.setColorDepth(16);
  lcd.setFont(&FreeSans12pt7b);

  // Set FSM initial state
  state = STATE_INIT;
}

/************************************************************************/
/*                             Program Loop                             */
/************************************************************************/

void loop() {

  // Check wifi connectivity once a minute
  if (futureTime < millis()) {
    futureTime = millis() + 60000;

    // Is WiFi still connected ?
    if ((WiFi.status() != WL_CONNECTED) || (!checkInternetConnection())) {

      WiFi.disconnect();
      WiFi.reconnect();

      Serial.printf("WiFi disconnected, trying to reconnect\n");

      // Connection dropped so restart ESP32
      // \esp_restart();
    } else {
      Serial.println("WiFi connected");
    }
  }

  // Get local time taking timezone and DST into consideratupdateTion
  uint32_t localTime = myTZ.toLocal(now());

  int today = day(localTime);
  int hours24 = hour(localTime);
  int hours12 = hourFormat12(localTime);
  bool am = isAM(localTime);
  int minutes = minute(localTime);
  int seconds = second(localTime);

  // Run the FSM
  switch (state) {

    case STATE_INIT:
      {
        // Turn backlight on
        backlight(true);

        // Clear screen to black
        lcd.clearDisplay(BLACK);

        // Set default text color
        lcd.setTextColor(WHITE);

        // Initialize variables
        prevSecond = 0;
        prevMinute = 0;
        prevDay = 0;

        // Draw static clock face
        drawClockFace();

        // Draw label at bottom under calendar
        int len = lcd.textWidth(BOTTOM_LABEL);
        int xOffset = ((CALENDAR_X_OFFSET - len) / 2) + CALENDAR_X_OFFSET;
        int yOffset = lcd.height() - 30;
        lcd.setCursor(xOffset, yOffset);
        lcd.print(BOTTOM_LABEL);

        // Set next state
        state = STATE_RUN;
      }
      break;

    case STATE_RUN:
      {
        // Check to see if clock should be on
        if ((hours24 > 5) and (hours24 < 22)) {
          // Clock should be on
          if (seconds != prevSecond) {
            prevSecond = seconds;
            updateTime(hours12, minutes, seconds);
          }
          if (minutes != prevMinute) {
            prevMinute = minutes;
            displayDigitalTime(hours12, minutes, am);
          }
          if (today != prevDay) {
            prevDay = today;
            drawCalendar(localTime);
          }
        } else {
          // Clock needs to go off
          // Set next state
          state = STATE_TURN_OFF;
        }
      }
      break;

    case STATE_TURN_OFF:
      {
        // Clear screen to black
        lcd.clearDisplay(BLACK);

        // Turn backlight off
        backlight(false);

        // Set next state
        state = STATE_WAIT;
      }
      break;

    case STATE_WAIT:
      {
        // Delay a minute
        delay(60000);

        // Check to see if clock should be on
        if ((hours24 > 5) and (hours24 < 22)) {

          // Set next state
          state = STATE_INIT;
        }
      }
      break;
  }
}
