/*
  ESP32 GPS Program using the Cheap Yellow Display

  Hardware Used:
    Cheap Yellow Display ESP32-2432S028R
    NEO-6M GPS Module with external antenna
  
  Libraries Used:
    Adafruit_GFX by Adafruit version 1.11.11
    Adafruit_ILI9341 by Adafruit version 1.6.1
    TinyGPS++ by Mikal Hart version 1.0.3
    Time by Michael Margolis version 1.6.1
    Timezone by Jack Christensen version 1.2.4

  Connections:
    The CN1 connector on the CYD is wired to the GPS module
    A serial interface is used between the CYD and the GPS
    for transferring data.
  
  Concept, Design and Implementation by: Craig A. Lindley
  Last Update: 12/17/2024
*/

#include <TinyGPS++.h>
#include <Time.h>
#include <Timezone.h>
#include <SPI.h>
#include <Adafruit_ILI9341.h>
#include <Adafruit_GFX.h>
#include <Fonts/FreeSans9pt7b.h>

#define APP_STR "ESP32  CYD  GPS  PROGRAM"
#define LOC_STR "-- LOCATION  DATA --"
#define WAIT_STR "Waiting for Satellites"
#define WAIT2_STR "(this could take awhile)"
#define AUT_STR "-- Craig A. Lindley 2024 --"

// Time between GPS updates
#define GPS_UPDATE_MINS 1
#define GPS_UPDATE_SECS (GPS_UPDATE_MINS * 60)
#define GPS_UPDATE_MS (GPS_UPDATE_SECS * 1000)

// Program variables
uint32_t futureTime;

// GPS data variables
double latitude;
double longitude;
double altitude;
double speed;
double hdop;
int satellites;

/************************************************************************/
/*                         Timezone properties                          */
/************************************************************************/

// This program runs in the Mountain Time Zone
// See Timezone.h for the details
#define DST_TIMEZONE_OFFSET -6  // Day Light Saving Time offset (-6 is mountain time)
#define ST_TIMEZONE_OFFSET -7   // Standard Time offset (-7 is mountain time)

// TimeZone and Daylight Savings Time Rules
// Define daylight savings time rules for the Mountain Time Zone
TimeChangeRule myDST = { "MDT", Second, Sun, Mar, 2, DST_TIMEZONE_OFFSET * 60 };
TimeChangeRule mySTD = { "MST", First, Sun, Nov, 2, ST_TIMEZONE_OFFSET * 60 };
Timezone myTZ(myDST, mySTD);

/**************************************************************/
/***                       GPS Parameters                   ***/
/**************************************************************/

// Define the RX and TX GPIO pins for Serial 2
// Note: transmit on the CYD is connected to RX on the GPS
//       receive  on the CYD is connected to TX on the GPS
// Note: Vcc (3.3V) is the red wire and Gnd is the black wire
#define RXD2 22  // Blue wire
#define TXD2 27  // Yellow wire

#define GPS_BAUD 9600

// Instantiate the TinyGPS++ object
TinyGPSPlus gps;

// Create an instance of the HardwareSerial class for Serial 2
HardwareSerial gpsSerial(2);

/**************************************************************/
/***                     Display Parameters                 ***/
/**************************************************************/

// Hardware connections from ESP32 to CYD
#define TFT_BL 21
#define TFT_CS 15
#define TFT_DC 2
#define TFT_MISO 12
#define TFT_MOSI 13
#define TFT_SCLK 14
#define TFT_RST -1

// Instantiate the CYD with the ILI9341 controller
Adafruit_ILI9341 tft = Adafruit_ILI9341(TFT_CS, TFT_DC, TFT_MOSI, TFT_SCLK, TFT_RST, TFT_MISO);

/**************************************************************/
/***                      Misc Functions                    ***/
/**************************************************************/

// Convert UTC time to local time
uint32_t utcTimeToLocalTime() {

  tmElements_t tm;

  // Get UTC time values
  uint16_t gpsUTCYear = gps.date.year();
  uint16_t gpsUTCMonth = gps.date.month();
  uint16_t gpsUTCDay = gps.date.day();
  uint16_t gpsUTCHour = gps.time.hour();

  // Advance minute if necessary
  uint16_t gpsUTCMinute = gps.time.minute();
  uint16_t gpsUTCSecond = gps.time.second();
  if (gpsUTCSecond > 30) {
    gpsUTCMinute++;
  }

  // Convert UTC to local time
  // First clear the structure
  memset(&tm, 0, sizeof(tmElements_t));

  // Fill the structure with UTC elements
  tm.Second = gpsUTCSecond;
  tm.Minute = gpsUTCMinute;
  tm.Hour = gpsUTCHour;
  tm.Day = gpsUTCDay;
  tm.Month = gpsUTCMonth;
  tm.Year = gpsUTCYear - 1970;

  // Turn UTC time into seconds since 1970
  time_t epochTime = makeTime(tm);

  // Return local time taking timezone and DST into consideration
  return myTZ.toLocal(epochTime);
}

// Print a centered string on specified lcd line
void printCenteredStringOnLine(int line, const char *str) {
  int16_t x1, y1;
  uint16_t w, h;

  tft.getTextBounds(str, 0, 0, &x1, &y1, &w, &h);
  tft.setCursor((320 - w) / 2, (line + 1) * 21);
  tft.print(str);
}

// Initial message while waiting
void initialUI() {

  // Clear the display
  tft.fillScreen(ILI9341_BLACK);

  printCenteredStringOnLine(0, APP_STR);
  printCenteredStringOnLine(5, WAIT_STR);
  printCenteredStringOnLine(6, WAIT2_STR);
  printCenteredStringOnLine(10, AUT_STR);
}

// Update the UI with possibly new values
void updateUI() {

  char buffer[128];

  int line = 0;

  // Clear the display
  tft.fillScreen(ILI9341_BLACK);

  printCenteredStringOnLine(line++, APP_STR);
  printCenteredStringOnLine(line++, LOC_STR);

  sprintf(buffer, "Lat: %.6f", latitude);
  printCenteredStringOnLine(line++, buffer);
  sprintf(buffer, "Lon: %.6f", longitude);
  printCenteredStringOnLine(line++, buffer);
  sprintf(buffer, "Alt: %.2f", altitude);
  printCenteredStringOnLine(line++, buffer);
  sprintf(buffer, "Speed: %.2f", speed);
  printCenteredStringOnLine(line++, buffer);
  sprintf(buffer, "HDOP: %.2f   Satellites: %d", hdop, satellites);
  printCenteredStringOnLine(line++, buffer);

  // Get local time
  uint32_t localTime = utcTimeToLocalTime();

  int curDOW = weekday(localTime);
  printCenteredStringOnLine(line++, dayStr(curDOW));

  int curDay = day(localTime);
  int curMon = month(localTime);
  int curYear = year(localTime);
  sprintf(buffer, "%s %d, %d", monthStr(curMon), curDay, curYear);
  printCenteredStringOnLine(line++, buffer);

  int theHour = hourFormat12(localTime);
  bool am = isAM(localTime);
  int theMinute = minute(localTime);

  // Add leading 0 to minute if necessary
  String minuteStr;
  if (theMinute < 10) {
    minuteStr = "0" + String(theMinute);
  } else {
    minuteStr = String(theMinute);
  }

  sprintf(buffer, "%d:%s %s", theHour, minuteStr.c_str(), am ? "AM" : "PM");
  printCenteredStringOnLine(line++, buffer);

  printCenteredStringOnLine(line++, AUT_STR);
}

/**************************************************************/
/***                       Program Setup                    ***/
/**************************************************************/

void setup() {

  Serial.begin(115200);
  delay(1000);
  Serial.println("Starting GPS");

  // Start Serial 2 with the defined RX and TX pins and a baud rate of 9600
  gpsSerial.begin(GPS_BAUD, SERIAL_8N1, RXD2, TXD2);
  Serial.println("Serial 2 started at 9600 baud rate");

  // Initialize display driver
  tft.begin();

  // Rotate the display
  tft.setRotation(3);

  // Set a font
  tft.setFont(&FreeSans9pt7b);

  // Set text color
  tft.setTextColor(ILI9341_WHITE);

  // Turn backlight on
  pinMode(TFT_BL, OUTPUT);
  digitalWrite(TFT_BL, HIGH);

  futureTime = 0;
  latitude = 0.0;
  longitude = 0.0;
  altitude = 0;
  speed = 0.0;
  hdop = 0.0;
  satellites = 0;

  // Display initial message
  initialUI();
}

/**************************************************************/
/***                       Program Loop                     ***/
/**************************************************************/

void loop(void) {

  // Look for new GPS data every minute
  if (millis() > futureTime) {
    futureTime = millis() + GPS_UPDATE_MS;

    // Attempt to read new GPS data if there is any
    while (gpsSerial.available() > 0) {
      gps.encode(gpsSerial.read());
    }

    // Has the GPS data been updated ?
    if (gps.location.isUpdated()) {
      latitude = gps.location.lat();
      longitude = gps.location.lng();
      speed = gps.speed.mph();
      altitude = gps.altitude.feet();
      hdop = gps.hdop.value();
      satellites = gps.satellites.value();

      // Update the UI with the new values
      updateUI();
    }
  }
}
